--[[
	Ezlo WebSocket client for Lua
	
	useage: Ezlo_ws_client.lua HubIP userID password
	
	Thanks to rigpapa for WebSocket module.
	
	V1.1, local mode only
	
	V1.1, use of nixio and faster sha1 code.
]]--

local dkjson 	= require("dkjson")
local cjson = nil
if pcall(require, "cjson") then
	cjson = require("cjson")
end	
local socket = require("socket")

-- Serial of Hub
local PK_AccessPoint = ""
local BuildVersion                -- ...of remote machine
local LoadTime                    -- ... ditto
-- Storage for local access credentials between sessions.
local ezlo_keys_store = ".ezlo_local_keys_lua"


local function WebSocketAPI()
--[[
	luws.lua - Luup WebSocket implemented (for Vera Luup and openLuup systems)
	Copyright 2020 Patrick H. Rigney, All Rights Reserved. http://www.toggledbits.com/
	Works best with SockProxy installed.
	Ref: RFC6455

	NOTA BENE: 64-bit payload length not supported.

RB: 	fix for messages larger than 256 bytes.
	fix for handling ping request
	removed chat option from negotiate.

--]]
--luacheck: std lua51,module,read globals luup,ignore 542 611 612 614 111/_,no max line length

--module("luws", package.seeall)

_VERSION = 20140

debug_mode = true

local math = require "math"
local string = require "string"
local socket = require "socket"
local bit = require "bit"
local lfs = require "lfs"
-- local ltn12 = require "ltn12"

-- local WSGUID = "258EAFA5-E914-47DA-95CA-C5AB0DC85B11"
local STATE_START = "start"
local STATE_READLEN1 = "len"
local STATE_READLEN161 = "len16-1"
local STATE_READLEN162 = "len16-2"
local STATE_READDATA = "data"
local STATE_SYNC = "sync"
-- local STATE_RESYNC1 = "resync1"
-- local STATE_RESYNC2 = "resync2"
local STATE_READMASK = "mask"
local MAXMESSAGE = 65535 -- maximum WS message size
--local CHUNKSIZE = 2048 -- Athom seems to have similar buffer size.
local CHUNKSIZE = 2048*8+8
local DEFAULTMSGTIMEOUT = 0 -- drop connection if no message in this time (0=no timeout)

local timenow = socket.gettime or os.time -- use hi-res time if available
local unpack = unpack or table.unpack -- luacheck: ignore 143
local LOG = function(msg,level)
	local now = timenow()
	print(os.date("%H:%M:%S",math.floor(now)).."."..math.floor((now - math.floor(now))*10000),msg) 
end

function dump(t, seen)
	if t == nil then return "nil" end
	if seen == nil then seen = {} end
	local sep = ""
	local str = "{ "
	for k,v in pairs(t) do
		local val
		if type(v) == "table" then
			if seen[v] then val = "(recursion)"
			else
				seen[v] = true
				val = dump(v, seen)
			end
		elseif type(v) == "string" then
			if #v > 255 then val = string.format("%q", v:sub(1,252).."...")
			else val = string.format("%q", v) end
		elseif type(v) == "number" and (math.abs(v-os.time()) <= 86400) then
			val = tostring(v) .. "(" .. os.date("%x.%X", v) .. ")"
		else
			val = tostring(v)
		end
		str = str .. sep .. k .. "=" .. val
		sep = ", "
	end
	str = str .. " }"
	return str
end

local function L(msg, ...) -- luacheck: ignore 212
	local str
	local level = 50
	if type(msg) == "table" then
		str = "luws: " .. tostring(msg.msg or msg[1])
		level = msg.level or level
	else
		str = "luws: " .. tostring(msg)
	end
	str = string.gsub(str, "%%(%d+)", function( n )
			n = tonumber(n, 10)
			if n < 1 or n > #arg then return "nil" end
			local val = arg[n]
			if type(val) == "table" then
				return dump(val)
			elseif type(val) == "string" then
				return string.format("%q", val)
			elseif type(val) == "number" and math.abs(val-os.time()) <= 86400 then
				return tostring(val) .. "(" .. os.date("%x.%X", val) .. ")"
			end
			return tostring(val)
		end
	)
	LOG(str, level)
end

local function D(msg, ...) if debug_mode then L( { msg=msg, prefix="luws[debug]: " }, ... ) end end

local function default( val, dflt ) return ( val == nil ) and dflt or val end

local function split( str, sep )
	sep = sep or ","
	local arr = {}
	if str == nil or #str == 0 then return arr, 0 end
	local rest = string.gsub( str or "", "([^" .. sep .. "]*)" .. sep, function( m ) table.insert( arr, m ) return "" end )
	table.insert( arr, rest )
	return arr, #arr
end

-- Upgrade an HTTP socket to websocket
local function wsupgrade( wsconn )
	D("wsupgrade(%1)", wsconn)
	local mime = require "mime"

	-- Generate key/nonce, 16 bytes base64-encoded
	local key = {}
	for k=1,16 do key[k] = string.char( math.random( 0, 255 ) ) end
	key = mime.b64( table.concat( key, "" ) )
	-- Ref: https://stackoverflow.com/questions/18265128/what-is-sec-websocket-key-for
	local req = string.format("GET %s HTTP/1.1\r\nHost: %s\r\nUpgrade: websocket\r\nConnection: Upgrade\r\nSec-WebSocket-Key: %s\r\nSec-WebSocket-Version: 13\r\n\r\n",
		wsconn.path, wsconn.ip, key)

	-- Send request.
	D("wsupgrade() sending %1", req)
	wsconn.socket:settimeout( 5, "b" )
	wsconn.socket:settimeout( 5, "r" )
	local nb,err = wsconn.socket:send( req )
	if nb == nil then
		return false, "Failed to send upgrade request: "..tostring(err)
	end

	-- Read until we get two consecutive linefeeds.
	wsconn.socket:settimeout( 5, "b" )
	wsconn.socket:settimeout( 5, "r" )
	local buf = {}
	local ntotal = 0
	while true do
		nb,err = wsconn.socket:receive("*l")
		D("wsupgrade() received %1, %2", nb, err)
		if nb == nil then
			L("wsupgrade() error while reading upgrade response, %1", err)
			return false
		end
		if #nb == 0 then break end -- blank line ends
		table.insert( buf, nb )
		ntotal = ntotal + #nb
		if ntotal >= MAXMESSAGE then
			buf = {}
			L({level=1,msg="Buffer overflow reading websocket upgrade response; aborting."})
			break;
		end
	end

	-- Check response
	-- local resp = table.concat( buf, "\n" )
	D("wsupdate() upgrade response: %1", buf)
	if buf[1]:match( "^HTTP/1%.. 101 " ) then
		-- ??? check response key TO-DO
		D("wsupgrade() upgrade succeeded!")
		wsconn.readstate = STATE_START
		return true -- Flag now in websocket protocol
	end
	return false, "upgrade failed; "..tostring(buf[1])
end

local function connect( ip, port )
	local sock = socket.tcp()
	if not sock then
		return nil, "Can't get socket for connection"
	end
	sock:settimeout( 15 )
	local r, e = sock:connect( ip, port )
	if r then
		return sock
	end
	sock:close()
	return nil, string.format("Connection to %s:%s failed: %s", ip, port, tostring(e))
end

function wsopen( url, handler, options )
	D("wsopen(%1)", url)
	options = options or {}
	options.receive_timeout = default( options.receive_timeout, DEFAULTMSGTIMEOUT )
	options.receive_chunk_size = default( options.receive_chunk_size, CHUNKSIZE )
	options.max_payload_size = default( options.max_payload_size, MAXMESSAGE )
	options.use_masking = default( options.use_masking, true ) -- RFC required, but settable
	options.connect = default( options.connect, connect )
	options.control_handler = default( options.control_handler, nil )

	local port
	local proto, ip, ps = url:match("^(wss?)://([^:/]+)(.*)")
	if not proto then
		error("Invalid protocol/address for WebSocket open in " .. url)
	end
	port = proto == "wss" and 443 or 80
	local p,path = ps:match("^:(%d+)(.*)")
	if p then
		port = tonumber(p) or port
	else
		path = ps
	end
	if path == "" then path = "/" end

	local wsconn = {}
	wsconn.connected = false
	wsconn.proto = proto
	wsconn.ip = ip
	wsconn.port = port
	wsconn.path = path
	wsconn.readstate = STATE_START
	wsconn.msg = nil
	wsconn.msghandler = handler
	wsconn.options = options

	-- This call is async -- it returns immediately.
	--??? options.create? for extensible socket creation?
	local sock,err = options.connect( ip, port )
	if not sock then
		return false, err
	end
	wsconn.socket = sock
	wsconn.socket:setoption( 'keepalive', true )
	if proto == "wss" then
		local ssl = require "ssl"
		local opts = {
			mode=default( options.ssl_mode, 'client' ),
			protocol=default( options.ssl_protocol, 'any' ),
			verify=default( options.ssl_verify, 'none' ),
			options=split( options.ssl_options, 'all' )
		}
		sock = ssl.wrap( wsconn.socket, opts )
		if sock and sock:dohandshake() then
			D("wsopen() successful SSL/TLS negotiation")
			wsconn.socket = sock -- save wrapped socket
		else
			wsconn.socket:close()
			wsconn.socket = nil
			return false, "Failed SSL negotation"
		end
	end
	D("wsopen() upgrading connection to WebSocket")
	st,err = wsupgrade( wsconn )
	if st then
		wsconn.connected = true
		wsconn.lastMessage = timenow()
		local m = getmetatable(wsconn) or {}
		m.__tostring = function( o ) return string.format("luws-websock[%s:%s]", o.ip, o.port) end
		-- m.__newindex = function( o, n, v ) error("Immutable luws-websock, can't set "..n) end
		setmetatable(wsconn, m)
		D("wsopen() successful WebSocket startup, wsconn %1", wsconn)
		return wsconn
	end
	wsconn.socket:close()
	wsconn.socket = nil
	return false, err
end

local function send_frame( wsconn, opcode, fin, s )
	D("send_frame(%1,%2,%3,<%4 bytes>)", wsconn, opcode, fin, #s)
	local mask = wsconn.options.use_masking
--	if opcode == 0x0a or opcode == 0x09 then mask = false end	-- Do not mask ping or pong response.
	local t = {}
	local b = bit.bor( fin and 0x80 or 0, opcode )
	table.insert( t, string.char(b) )
	if #s < 126 then
		table.insert( t, string.char(#s + ( mask and 128 or 0)) )
	elseif #s < 65536 then
		table.insert( t, string.char(126 + ( mask and 128 or 0)) ) -- indicate 16-bit length follows
		table.insert( t, string.char( math.floor( #s / 256 ) ) )
		table.insert( t, string.char( #s % 256 ) )
	else
		-- We don't currently support 64-bit frame length (caller shouldn't be trying, either)
		error("Super-long frame length not implemented")
	end
	local frame
	if mask then
		-- Generate mask and append to frame.
		local mb = { 0,0,0,0 }
		for k=1,4 do
			mb[k] = math.random(0,255)
			table.insert( t, string.char( mb[k] ) )
		end
		D("send_frame() mask bytes %1", string.format( "%02x %02x %02x %02x", mb[1], mb[2], mb[3], mb[4] ) )
		-- Apply mask to data and append.
		for k=1,#s do
			table.insert( t, string.char( bit.bxor( string.byte( s, k ), mb[((k-1)%4)+1] ) ) )
		end
		frame = table.concat( t, "" )
	else
		-- No masking, just concatenate string as we got it (not RFC for client).
		frame = table.concat( t, "" ) .. s
	end
	t = nil -- luacheck: ignore 311
	D("send_frame() sending frame of %1 bytes for %2", #frame, s)
	wsconn.socket:settimeout( 5, "b" )
	wsconn.socket:settimeout( 5, "r" )
	-- ??? need retry while nb < payload length
	while true do
		local nb,err = wsconn.socket:send( frame )
		if not nb then return false, "send error: "..tostring(err) end
		if nb >= #frame then break end
		frame = frame:sub( nb + 1 )
	end
	return true
end

-- Send WebSocket message (opcode and payload). The payload can be an LTN12 source,
-- in which case each chunk from the source is sent as a fragment.
function wssend( wsconn, opcode, s )
	D("wssend(%1,%2,%3)", wsconn, opcode, s)
	if not wsconn.connected then return false, "not connected" end
	if wsconn.closing then return false, "closing" end

	if opcode == 0x08 then
		wsconn.closing = true -- sending close frame
	end

	if type(s) == "function" then
		-- A function as data is assumed to be an LTN12 source
		local chunk, err = s() -- get first chunk
		while chunk do
			local next_chunk, nerr = s() -- get another
			local fin = next_chunk == nil -- no more?
			assert( #chunk < 65536, "LTN12 source returned excessively long chunk" )
			send_frame( wsconn, opcode, fin, chunk ) -- send last
			opcode = 0 -- continuations from here out
			chunk,err = next_chunk, nerr -- new becomes last
		end
		return err == nil, err
	end

	-- Send as string buffer
	if type(s) ~= "string" then s = tostring(s) end
	if #s < 65535 then
		return send_frame( wsconn, opcode, true, s ) -- single frame
	else
		-- Long goes out in 64K-1 chunks; op + noFIN first, op0 + noFIN continuing, op0+FIN final.
		repeat
			local chunk = s:sub( 1, 65535 )
			s = s:sub( 65536 )
			local fin = #s == 0 -- fin when out of data (last chunk)
			if not send_frame( wsconn, opcode, fin, chunk ) then
				return false, "send error"
			end
			opcode = 0 -- all following fragments go as continuation
		until #s == 0
	end
	return true
end

-- Disconnect websocket interface, if connected (safe to call any time)
function wsclose( wsconn )
	D("wsclose(%1)", wsconn)
	if wsconn then
		-- This is not in keeping with the RFC, but may be as good as we can reliably do.
		-- We don't wait for a close reply, just send it and shut down.
		if wsconn.socket and wsconn.connected and not wsconn.closing then
			wsconn.closing = true
			wssend( wsconn, 0x08, "" )
		end
		if wsconn.socket then
			wsconn.socket:close()
			wsconn.socket = nil
			wsconn.connected = false
		end
	end
end

-- Handle a control frame. Caller is given option first.
local function handle_control_frame( wsconn, opcode, data )
	D("handle_control_frame(%1,%2,%3)", wsconn, opcode, data )
	if wsconn.options.control_handler and
		false == wsconn.options.control_handler( wsconn, opcode, data, unpack(wsconn.options.handler_args or {}) ) then
		-- If custom handler returns exactly boolean false, don't do default actions
		return
	end
	if opcode == 0x08 then -- close
		if not wsconn.closing then
			wsconn.closing = true
			wssend( wsconn, 0x08, "" )
		end
		-- Notify
		pcall( wsconn.msghandler, wsconn, false, "receiver error: closed", unpack(wsconn.options.handler_args or {}) )
	elseif opcode == 0x09 then -- ping
		wssend( wsconn, 0x0a, "" ) -- reply with pong
		wsconn.lastping_ts = timenow()
	else
		wsconn.lastping_ts = timenow()
		-- 0x0a pong, no action
		-- Other unsupported control frame
	end
end

-- Take incoming fragment and accumulate into message (or, maybe it's the whole
-- message, or a control message). Dispatch complete and control messages.
-- ??? best application for LTN12 here?
local function wshandlefragment( fin, op, data, wsconn )
	-- D("wshandlefragment(%1,%2,<%3 bytes>,%4)", fin, op, #data, wsconn)
	if fin then
		-- FIN frame
		wsconn.lastMessage = timenow()
		if op >= 8 then
			handle_control_frame( wsconn, op, data )
			return
		elseif (wsconn.msg or "") == "" then
			-- Control frame or FIN on first packet, handle immediately, no copy/buffering
			D("wshandlefragment() fast dispatch %1 byte message for op %2", #data, op)
			return pcall( wsconn.msghandler, wsconn, op, data, unpack(wsconn.options.handler_args or {}) )
		end
		-- Completion of continuation; RFC6455 requires final fragment to be op 0 (we tolerate same op)
		if op ~= 0 and op ~= wsconn.msgop then
			return pcall( wsconn.msghandler, wsconn, false, "ws completion error",
				unpack(wsconn.options.handler_args or {}) )
		end
		-- Append to buffer and send message
		local maxn = math.max( 0, wsconn.options.max_payload_size - #wsconn.msg )
		if maxn < #data then
			D("wshandlefragment() buffer overflow, have %1, incoming %2, max %3; message truncated.",
				#wsconn.msg, #data, wsconn.options.max_payload_size)
		end
		if maxn > 0 then 
			wsconn.msg = wsconn.msg .. data:sub(1, maxn) 
		end
		D("wshandlefragment() dispatch %2 byte message for op %1", wsconn.msgop, #wsconn.msg)
		wsconn.lastMessage = timenow()
		pcall( wsconn.msghandler, wsconn, wsconn.msgop, wsconn.msg, unpack(wsconn.options.handler_args or {}) )
		wsconn.msg = nil
	else
		-- No FIN
		if (wsconn.msg or "") == "" then
			-- First fragment, also save op (first determines for all)
			D("wshandlefragment() no fin, first fragment")
			wsconn.msgop = op
			wsconn.msg = data
		else
			D("wshandlefragment() no fin, additional fragment")
			-- RFC6455 requires op on continuations to be 0.
			if op ~= 0 then 
				return pcall( wsconn.msghandler, wsconn, false, "ws continuation error", unpack(wsconn.options.handler_args or {}) ) 
			end
			local maxn = math.max( 0, wsconn.options.max_payload_size - #wsconn.msg )
			if maxn < #data then
				L("wshandlefragment() buffer overflow, have %1, incoming %2, max %3; message truncated",
					#wsconn.msg, #data, wsconn.options.max_payload_size)
			end
			if maxn > 0 then wsconn.msg = wsconn.msg .. data:sub(1, maxn) end
		end
	end
end

-- Unmask buffered data fragments
local function unmask( fragt, maskt )
	local r = {}
	for _,d in ipairs( fragt or {} ) do
		for l=1,#d do
			local k = (#r % 4) + 1 -- convenient
			table.insert( r, string.char( bit.bxor( string.byte( d, l ), maskt[k] ) ) )
		end
	end
	return table.concat( r, "" )
end

-- Handle a block of data. The block does not need to contain an entire message
-- (or fragment). A series of blocks as small as one byte can be passed and the
-- message accumulated properly within the protocol.
function wshandleincoming( data, wsconn )
--	D("wshandleincoming(<%1 bytes>,%2) in state %3", #data, wsconn, wsconn.readstate)
	local state = wsconn
	local ix = 1
	while ix <= #data do
		local b = data:byte( ix )
		-- D("wshandleincoming() at %1/%2 byte %3 (%4) state %5", ix, #data, b, string.format("%02X", b), state.readstate)
		if state.readstate == STATE_READDATA then
			-- ??? WHAT ABOUT UNMASKING???
			-- Performance: this at top; table > string concatenation; handle more than one byte, too.
			-- D("wshandleincoming() read state, %1 bytes pending, %2 to go in message", #data, state.flen)
			local nlast = math.min( ix + state.flen - 1, #data )
			D("wshandleincoming() nlast is %1, length accepting %2", nlast, nlast-ix+1)
			table.insert( state.frag, data:sub( ix, nlast ) )
			state.flen = state.flen - ( nlast - ix + 1 )
			if debug_mode and state.flen % 500 == 0 then D("wshandleincoming() accepted, now %1 bytes to go", state.flen) end
			if state.flen <= 0 then
				local delta = math.max( timenow() - state.start, 0.001 )
				D("wshandleincoming() message received, %1 bytes in %2 secs, %3 bytes/sec, %4 chunks", state.size, delta, state.size / delta, #state.frag)
				local f = state.masked and unmask( state.frag, state.mask ) or table.concat( state.frag, "" )
				state.frag = nil -- gc eligible
				state.readstate = STATE_START -- ready for next frame
				wshandlefragment( state.fin, state.opcode, f, wsconn )
			end
			ix = nlast
		elseif state.readstate == STATE_START then
			D("wshandleincoming() start at %1 byte %2", ix, string.format("%02X", b))
			state.fin = bit.band( b, 128 ) > 0
			state.opcode = bit.band( b, 15 )
			state.flen = 0 -- remaining data bytes to receive
			state.size = 0 -- keep track of original size
			state.masked = nil
			state.mask = nil
			state.masklen = nil
			state.frag = {}
			state.readstate = STATE_READLEN1
			state.start = timenow()
			D("wshandleincoming() start of frame, opcode %1 fin %2", state.opcode, state.fin)
		elseif state.readstate == STATE_READLEN1 then
			state.masked = bit.band( b, 128 ) > 0
			state.flen = bit.band( b, 127 )
			if state.flen == 126 then
				-- Payload length in 16 bit integer that follows, read 2 bytes (big endian)
				state.readstate = STATE_READLEN161
			elseif state.flen == 127 then
				-- 64-bit length (unsupported, ignore message)
				L{level=2,msg="Ignoring 64-bit length frame, not supported"}
				state.readstate = STATE_SYNC
			else
				-- 7-bit payload length
				D("wshandleincoming() short length, expecting %1 byte payload", state.flen)
				state.size = state.flen
				if state.flen > 0 then
					-- Transition to reading data.
					state.readstate = state.masked and STATE_READMASK or STATE_READDATA
				else
					-- No data with this opcode, process and return to start state.
					wshandlefragment( state.fin, state.opcode, "", wsconn )
					state.readstate = STATE_START
				end
			end
			D("wshandleincoming() opcode %1 len %2 next state %3", state.opcode, state.flen, state.readstate)
		elseif state.readstate == STATE_READLEN161 then
			state.flen = b * 256
			state.readstate = STATE_READLEN162
		elseif state.readstate == STATE_READLEN162 then
			state.flen = state.flen + b
			state.size = state.flen
			state.readstate = state.masked and STATE_READMASK or STATE_READDATA
			D("wshandleincoming() finished 16-bit length read, expecting %1 byte payload", state.size)
		elseif state.readstate == STATE_READMASK then
			-- ??? According to RFC6455, we MUST error and close for masked data from server [5.1]
			if not state.mask then
				state.mask = { b }
			else
				table.insert( state.mask, b )
				if #state.mask >= 4 then
					state.readstate = STATE_READDATA
				end
			end
			D("wshandleincoming() received %1 mask bytes, now %2", state.masklen, state.mask)
		elseif state.readstate == STATE_SYNC then
			return pcall( state.msghandler, wsconn, false, "lost sync", unpack(wsconn.options.handler_args or {}) )
		else
			assert(false, "Invalid state in wshandleincoming: "..tostring(state.readstate))
		end
		ix = ix + 1
	end
	D("wshandleincoming() ending state is %1", state.readstate)
end

-- Receiver task. Use non-blocking read. Returns nil,err on error, otherwise true/false is the
-- receiver believes there may immediately be more data to process.
function wsreceive( wsconn )
--	D("wsreceive(%1)", wsconn)
	if not wsconn.connected then return end
	wsconn.socket:settimeout( 0, "b" )
	wsconn.socket:settimeout( 0, "r" )
	--[[ PHR 20140: Make sure we provide a number of bytes. Failing to do so apparently kicks-in the
	                special handling of special characters, including 0 bytes, CR, LF, etc. We want
	                the available data completely unmolested.
	--]]
	local nb,err,bb = wsconn.socket:receive( wsconn.options.receive_chunk_size or CHUNK_SIZE )
	if nb == nil then
		if err == "timeout" or err == "wantread" then
			if bb and #bb > 0 then
				D("wsreceive() %1; handling partial result %2 bytes", err, #bb)
				wshandleincoming( bb, wsconn )
				wsconn.lastping_ts = timenow()
				return false, #bb -- timeout, say no more data
			elseif wsconn.options.receive_timeout > 0 and
				( timenow() - wsconn.lastMessage ) > wsconn.options.receive_timeout then
				pcall( wsconn.msghandler, wsconn, false, "message timeout",
					unpack(wsconn.options.handler_args or {}) )
				return nil, "message timeout"
			end
			return false, 0 -- not error, no data was handled
		end
		-- ??? error
		pcall( wsconn.msghandler, wsconn, false, "receiver error: "..err, unpack(wsconn.options.handler_args or {}) )
		return nil, err
	end
	D("wsreceive() handling %1 bytes", #nb)
	if #nb > 0 then
		wshandleincoming( nb, wsconn )
		wsconn.lastping_ts = timenow()
	end
	return #nb > 0, #nb -- data handled, maybe more?
end

function wslastping(wsconn)
--	D("wslastping(%1)", wsconn)
	if not wsconn.lastping_ts then wsconn.lastping_ts = timenow() end
	return wsconn.lastping_ts
end

-- Reset receiver state. Brutal resync; may or may be usable, but worth having the option.
function wsreset( wsconn )
	D("wsreset(%1)", wsconn)
	if wsconn then
		wsconn.msg = nil -- gc eligible
		wsconn.frag = nil -- gc eligible
		wsconn.readstate = STATE_START -- ready for next frame
	end
end

	local function init(debug_flg)
		debug_mode = debug_flg
	end

	return {
		Initialize = init,
		wsopen = wsopen,
		wsclose = wsclose,
		wssend = wssend,
		wsreceive = wsreceive,
		wsreset = wsreset,
		wslastping = wslastping
	}
end

-- Wrapper for more solid handling for cjson as it trows a bit more errors that I'd like.
-- cjson handled null value different. Not using for now
local function jsonAPI()
local is_cj, is_dk = false, false

	local function _init()
		is_cj = type(cjson) == "table"
		is_dk = type(dkson) == "table"
	end
	
	local function _decode(data)
		if is_cj then
			local ok, res = pcall(cjson.decode, data)
			if ok then return res end
		end
		local res, pos, msg = dkjson.decode(data)
		return res, msg
	end
	
	local function _encode(data)
		-- No special chekcing required as we must pass valid data our selfs
		if is_cj then
			return cjson.encode(data)
		else
			return dkjson.encode(data)
		end
	end
	
	return {
		Initialize = _init,
		decode = _decode,
		encode = _encode,
	}
end
local json = jsonAPI()

-- API to handle basic logging and debug messaging
local function logAPI()
local def_level = 1
local def_prefix = ""
local def_debug = false
local def_file = false
local max_length = 100
local onOpenLuup = false
local taskHandle = -1

	local function _update(level)
		if type(level) ~= "number" then level = def_level end
		if level >= 100 then
			def_file = true
			def_debug = true
			def_level = 10
		elseif level >= 10 then
			def_debug = true
			def_file = false
			def_level = 10
		else
			def_file = false
			def_debug = false
			def_level = level
		end
	end	

	local function _init(prefix, level, onol)
		_update(level)
		def_prefix = prefix
		onOpenLuup = onol
	end	
	local function _ts()
		local now = socket.gettime()
		return os.date("%H:%M:%S",math.floor(now)).."."..math.floor((now - math.floor(now))*10000)
	end
	
	-- Build loggin string safely up to given lenght. If only one string given, then do not format because of length limitations.
	local function prot_format(ln,str,...)
		local msg = ""
		local sf = string.format
		if arg[1] then 
			_, msg = pcall(sf, str, unpack(arg))
		else 
			msg = str or "no text"
		end 
		if ln > 0 then
			return msg:sub(1,ln)
		else
			return msg
		end	
	end	
	local function _log(...) 
		if (def_level >= 10) then
			print(_ts().." "..def_prefix .. ": " .. prot_format(max_length,...), 50) 
		end	
	end	
	
	local function _info(...) 
		if (def_level >= 8) then
			print(_ts().." "..def_prefix .. "_info: " .. prot_format(max_length,...), 8) 
		end	
	end	

	local function _warning(...) 
		if (def_level >= 2) then
			print(_ts().." "..def_prefix .. "_warning: " .. prot_format(max_length,...), 2) 
		end	
	end	

	local function _error(...) 
		if (def_level >= 1) then
			print(_ts().." "..def_prefix .. "_error: " .. prot_format(max_length,...), 1) 
		end	
	end	

	local function _debug(...)
		if def_debug then
			print(_ts().." "..def_prefix .. "_debug: " .. prot_format(-1,...), 50) 
		end	
	end
	
	
	return {
		Initialize = _init,
		Error = _error,
		Warning = _warning,
		Info = _info,
		Log = _log,
		Debug = _debug,
		Update = _update
	}
end 
local log = logAPI()

-- API for Ezlo Communications
local function ezloAPI()
	local ltn12 	= require("ltn12")
	local https     = require("ssl.https")
	local bit 		= require("bit")
	local nixio = nil
	if pcall(require, "nixio") then
		-- On Vera, use nixio crypto and b64decode module
		nixio = require("nixio")
	end

	local ezloPort = "17000"
	local maxReconnectRetries = 30	-- Allow for 15 minute reconnect retry. Should be plenty for reboot.
	local reconnectRetryInterval = 30
	local wssToken = nil
	local wssUser = nil
	local STAT = {
		CONNECT_FAILED = -4,
		BAD_PASSWORD = -3,
		TOKEN_EXPIRED = -2,
		NO_CONNECTION = -1,
		CONNECTING = 0,
		CONNECTED = 2,
		IDLE = 3,
		BUSY = 4
	}
	local connectionsStatus = STAT.NO_CONNECTION
	local wsconn = nil
	local hubIp = nil
	local methodCallbacks = {}
	local broadcastCallbacks = {}
	local errorCallbacks = {}
	local pingCounter = 0
	local pingCommand = nil	-- Send this data instead of Ping
	local luaws = WebSocketAPI()

	-- Calculates SHA1 for a string, returns it encoded as 40 hexadecimal digits.
	local function sha1(str)
		if nixio then
			-- Vera 
			local crypto = nixio.crypto.hash ("sha1")
			crypto = crypto.update(crypto, str)
			local hex, buf = crypto.final(crypto)
			return hex
		else
			-- Other
			local brol = bit.rol
			local band = bit.band
			local bor = bit.bor
			local bxor = bit.bxor
			local uint32_lrot = brol
			local uint32_xor_3 = bxor
			local uint32_xor_4 = bxor
			local sbyte = string.byte
			local schar = string.char
			local sformat = string.format
			local srep = string.rep

			local function uint32_ternary(a, b, c)
				-- c ~ (a & (b ~ c)) has less bitwise operations than (a & b) | (~a & c).
				return bxor(c, band(a, bxor(b, c)))
			end

			local function uint32_majority(a, b, c)
				-- (a & (b | c)) | (b & c) has less bitwise operations than (a & b) | (a & c) | (b & c).
				return bor(band(a, bor(b, c)), band(b, c))
			end

			-- Merges four bytes into a uint32 number.
			local function bytes_to_uint32(a, b, c, d)
				return a * 0x1000000 + b * 0x10000 + c * 0x100 + d
			end

			-- Splits a uint32 number into four bytes.
			local function uint32_to_bytes(a)
				local a4 = a % 256
				a = (a - a4) / 256
				local a3 = a % 256
				a = (a - a3) / 256
				local a2 = a % 256
				local a1 = (a - a2) / 256
				return a1, a2, a3, a4
			end

			local function hex_to_binary(hex)
				return (hex:gsub("..", function(hexval)
					return schar(tonumber(hexval, 16))
				end))
			end

			-- Input preprocessing.
			-- First, append a `1` bit and seven `0` bits.
			local first_append = schar(0x80)

			-- Next, append some zero bytes to make the length of the final message a multiple of 64.
			-- Eight more bytes will be added next.
			local non_zero_message_bytes = #str + 1 + 8
			local second_append = srep(schar(0), -non_zero_message_bytes % 64)

			-- Finally, append the length of the original message in bits as a 64-bit number.
			-- Assume that it fits into the lower 32 bits.
			local third_append = schar(0, 0, 0, 0, uint32_to_bytes(#str * 8))
			str = str .. first_append .. second_append .. third_append
			assert(#str % 64 == 0)

			-- Initialize hash value.
			local h0 = 0x67452301
			local h1 = 0xEFCDAB89
			local h2 = 0x98BADCFE
			local h3 = 0x10325476
			local h4 = 0xC3D2E1F0
			local w = {}

			-- Process the input in successive 64-byte chunks.
			for chunk_start = 1, #str, 64 do
				-- Load the chunk into W[0..15] as uint32 numbers.
				local uint32_start = chunk_start
				for i = 0, 15 do
					w[i] = bytes_to_uint32(sbyte(str, uint32_start, uint32_start + 3))
					uint32_start = uint32_start + 4
				end
				-- Extend the input vector.
				for i = 16, 79 do
					w[i] = uint32_lrot(uint32_xor_4(w[i - 3], w[i - 8], w[i - 14], w[i - 16]), 1)
				end
				-- Initialize hash value for this chunk.
				local a = h0
				local b = h1
				local c = h2
				local d = h3
				local e = h4
				-- Main loop.
				for i = 0, 79 do
					local f
					local k
					if i <= 19 then
						f = uint32_ternary(b, c, d)
						k = 0x5A827999
					elseif i <= 39 then
						f = uint32_xor_3(b, c, d)
						k = 0x6ED9EBA1
					elseif i <= 59 then
						f = uint32_majority(b, c, d)
						k = 0x8F1BBCDC
					else
						f = uint32_xor_3(b, c, d)
						k = 0xCA62C1D6
					end
					local temp = (uint32_lrot(a, 5) + f + e + k + w[i]) % 4294967296
					e = d
					d = c
					c = uint32_lrot(b, 30)
					b = a
					a = temp
				end
				-- Add this chunk's hash to result so far.
				h0 = (h0 + a) % 4294967296
				h1 = (h1 + b) % 4294967296
				h2 = (h2 + c) % 4294967296
				h3 = (h3 + d) % 4294967296
				h4 = (h4 + e) % 4294967296
			end
			return sformat("%08x%08x%08x%08x%08x", h0, h1, h2, h3, h4)
		end
	end
	
	-- Generate a (semi) random UUID
	local function uuid()
		local random = math.random
		local template ='xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'
		return string.gsub(template, '[xy]', function (c)
			local v = (c == 'x') and random(0, 0xf) or random(8, 0xb)
			return string.format('%x', v)
		end)
	end

	-- Base 64 decoding
	local function b64decode(data)
		if nixio then
			return nixio.bin.b64decode(data)
		else
			local b='ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'
			data = string.gsub(data, '[^'..b..'=]', '')
			return (data:gsub('.', function(x)
				if (x == '=') then return '' end
				local r,f='',(b:find(x)-1)
				for i=6,1,-1 do r=r..(f%2^i-f%2^(i-1)>0 and '1' or '0') end
				return r;
				end):gsub('%d%d%d?%d?%d?%d?%d?%d?', function(x)
					if (#x ~= 8) then return '' end
					local c=0
					for i=1,8 do c=c+(x:sub(i,i)=='1' and 2^(8-i) or 0) end
					return string.char(c)
				end))
		end
	end
	
	-- Generate a semi random message ID value
	local function generateID(prefix)
		local prefix = prefix or ""
		local random = math.random
		local template ='xxxxxxxxxxxxxx'
		return prefix..string.gsub(template, '[xy]', function (c)
			local v = (c == 'x') and random(0, 0xf) or random(8, 0xb)
			return string.format('%x', v)
		end)
	end	
	-- Close connection to Hub
	local function Close()
		luaws.wsclose(wsconn)
		wsconn = nil
		connectionsStatus = STAT.NO_CONNECTION
	end
	

	-- Handle WebSocket incomming messages
	local function MessageHandler(conn, opcode, data, ...)
		log.Debug("MessageHandler %s, %s",tostring(opcode), tostring(data))
		if opcode == 0x01 then
			-- Received text data, should be json to decode
			local js, msg = json.decode(data)
			-- Check for error table to be present (cannot test for nil as cjson returns userdata type)
			if type(js.error) == "table" then
				log.Debug("MessageHandler, response has error. %s, %s.", js.error.code, js.error.message)
				local func = errorCallbacks[js.method] or errorCallbacks["*"]
				if func then
					-- Call the registered handler
					local stat, msg = pcall(func, js.method, js.error, js.result)
					if not stat then
						log.Error("Error in error callback for method %s, msg %s",js.method,msg)
					end
				else
					-- No call back
					log.Debug("No error callback for method %s.",js.method)
				end	
				return
			elseif js.method ~= nil then
				-- look at method to handle. The Hub replies the method sent in the response
				if js.method == "hub.offline.login.ui" then
					-- Local logon completed, flag it
					log.Debug("MessageHandler, logon complete. Ready for commands.")
					connectionsStatus = STAT.CONNECTED
				end
				local func = methodCallbacks[js.method] or methodCallbacks["*"]
				if func then
					-- Call the registered handler
					local stat, msg = pcall(func, js.method, js.result)
					if not stat then
						log.Error("Error in method callback for method %s, msg %s",js.method,msg)
					end
				else
					-- No call back
					log.Debug("No method callback for method %s.",js.method)
				end	
			elseif js.id == "ui_broadcast" and js.msg_subclass ~= nil then
				local func = broadcastCallbacks[js.msg_subclass] or broadcastCallbacks["*"]
				if func then
					-- Call the registered handler
					local stat, msg = pcall(func, js.msg_subclass, js.result)
					if not stat then
						log.Error("Error in broadcast callback for message %s, msg %s",js.msg_subclass,msg)
					end
				else
					-- No call back
					log.Debug("No broadcast callback for message %s.",js.msg_subclass)
				end	
				return
			else
				log.Debug("MessageHandler, response has no method. Cannot process.")
			end
		elseif opcode == 0x02 then
			-- Received binary data. Not expecting this.
			log.Debug("MessageHandler, received binary data. Cannot process.")
		elseif opcode == 0x09 then
			-- Received ping (should be handled by luaws)
			log.Debug("MessageHandler, received ping.")
		elseif opcode == 0x0a then
			-- Received pong (should not be possibe)
			log.Debug("MessageHandler, received pong.")
		elseif opcode == 0x08 then
			-- Close by peer
			log.Debug("MessageHandler, Connection is closed by Hub.")
			Close()
		elseif opcode == false then
			log.Info("MessageHandler, opcode = false? %s.", tostring(data))
		else
			log.Error("MessageHandler, Unknown opcode.")
			--WebSocketConnectionOpen = -1
		end
	end
	
	-- Logon to Ezlo portal and return 
	local function PortalLogin(user_id, password, serial)
		local Ezlo_MMS_salt = "oZ7QE6LcLJp6fiWzdqZc"
		local authentication_url = "https://vera-us-oem-autha11.mios.com/autha/auth/username/%s?SHA1Password=%s&SHA1PasswordCS=%s&PK_Oem=1&TokenVersion=2"
		local get_token_url = "https://cloud.ezlo.com/mca-router/token/exchange/legacy-to-cloud/"
		local sync_token_url = "https://api-cloud.ezlo.com/v1/request"

		-- Do https request. For json requests and response data only.
		local function https_request(mthd, strURL, headers, PostData)
			local result = {}
			local request_body = nil
			if PostData then
				request_body=json.encode(PostData)
				headers["content-length"] = string.len(request_body)
			else
				headers["content-length"] = "0"
			end
log.Debug(strURL)			
			local bdy,cde,hdrs,stts = https.request{
				url = strURL, 
				method = mthd,
				sink = ltn12.sink.table(result),
				source = ltn12.source.string(request_body),
				protocol = "any",
				options =  {"all", "no_sslv2", "no_sslv3"},
				verify = "none",
				headers = headers
			}
			if bdy == 1 then
				if cde ~= 200 then
					return false, cde, nil, stts
				else
log.Debug(table.concat(result))				
					return true, cde, dkjson.decode(table.concat(result)), "OK"
				end
			else
				-- Bad request
				return false, 400, nil, "HTTP/1.1 400 BAD REQUEST"
			end
		end	

		-- Get Tokens
		local request_headers = {
			["access-control-allow-origin"] = "*",
			["user-agent"] = "RB Ezlo Bridge 1.0",
			["accept"] = "application/json",
			["content-type"] = "application/json; charset=UTF-8"
		}
		local SHA1pwd = sha1(string.lower(user_id)..password..Ezlo_MMS_salt)
		local SHA1pwdCS = sha1(user_id..password..Ezlo_MMS_salt)
		local stat, cde, response, msg = https_request("GET", authentication_url:format(user_id,SHA1pwd,SHA1pwdCS), request_headers)
		if not stat then
			log.Debug("Portal logon failed %s, %s",cde,msg)
			return false, 'Could not logon to portal'
		end	
		local MMSAuth = response.Identity
		local MMSAuthSig = response.IdentitySignature
		-- Identity has base64 encoded account details.
		local js_Ident = json.decode(b64decode(MMSAuth))
log.Debug("MMSAuthSig : %s", MMSAuthSig)		
log.Debug("MMSAuth : %s", b64decode(MMSAuth))		
		token_expires = js_Ident.Expires -- Need to logon again when token has expired.
		log.Debug(os.date("Token expires at : %c", token_expires))
		request_headers["MMSAuth"] = MMSAuth
		request_headers["MMSAuthSig"] = MMSAuthSig
		stat, cde, response, msg = https_request("GET", get_token_url, request_headers)
		if not stat then
			log.Debug("Get Token failed %s, %s",cde,msg)
			return false, 'Could not get token'
		end	
	
		-- Get controller keys (user & token)
log.Debug("Token : %s",response.token)
		local post_headers = {
			["authorization"] = "Bearer "..response.token,
			["access-control-allow-origin"] = "*",
			["user-agent"] = "RB Ezlo Bridge 1.0",
			["accept"] = "application/json",
			["content-type"] = "application/json; charset=UTF-8"
		}
		local post_data = {
			["call"] = "access_keys_sync",
			["version"] = "1",
			["params"] = {
				["version"] = 53, 
				["entity"] = "controller",
				["uuid"] = uuid()
			}
		}
		local stat, cde, response, msg = https_request("POST",sync_token_url, post_headers, post_data)
		if not stat then
			log.Debug("Sync controller keys failed %s, %s",cde,msg)
			return false, 'Could not sync controller keys'
		end	
		-- Get user and token from response.
		data = response.data
		local wss_user = ''
		local wss_token = ''
		local contr_uuid = ''
		for key, key_data in pairs(data.keys) do
			if key_data.meta then
				if key_data.meta.entity then
					if key_data.meta.entity.id then
						if key_data.meta.entity.id == serial then
							contr_uuid = key_data.meta.entity.uuid
						end
					end
				end
			end
		end
		if contr_uuid == '' then
			return false, "Controller serial not found"
		end
		for key, key_data in pairs(data.keys) do
			if key_data.data and wss_user == '' and wss_token == '' then
				if key_data.data.string then
					if key_data.meta.target.uuid == contr_uuid then
						wss_token = key_data.data.string
						wss_user = key_data.meta.entity.uuid
					end
				end
			end
		end
		if wss_user ~= '' and wss_token ~= '' then
			return true, wss_token, wss_user, token_expires
		else
			return false, "Could not obtain token data."
		end
	end

	-- Send text data to Hub. Cannot do simple json encode as with no params that will generate "params":[] and the G150 does not like that.
	local function Send(data)
		if connectionsStatus ~= STAT.CONNECTING and connectionsStatus ~= STAT.CONNECTED then
			log.Debug("No connection when trying to call Send()")
			return false, "No connection"
		end
		local id = generateID()
		local params = "{}"
		if data.params then
			if type(data.params) == "string" then
				params = data.params
			else
				params = dkjson.encode(data.params) 
			end
		end
		local cmd = '{"method":"%s","id":"%s","params":%s}'
log.Debug("sending command : "..(cmd:format(data.method, id, params) or "fail"))
		return luaws.wssend(wsconn, 0x01, cmd:format(data.method, id, params))
	end

	-- Non-blocking Read from Hub. Responses will be handled by MessageHandler
	local function Receive()
		return luaws.wsreceive(wsconn)
	end

	-- Send ping to Hub. 
	local function Ping()
		return luaws.wssend(wsconn, 0x09, "")
	end

	-- open web socket connection
	local function Connect(controller_ip, wss_token, wss_user)
		-- Use if hub.offline.insecure_access.enabled.set true
--		wsconn, msg = luaws.wsopen('ws://' .. controller_ip .. ':' .. ezloPort, MessageHandler)
		-- Use if hub.offline.insecure_access.enabled.set false
		wsconn, msg = luaws.wsopen('wss://' .. controller_ip .. ':' .. ezloPort, MessageHandler)
		if wsconn == false then
			return false, "Could not open WebSocket. " .. tostring(msg or "")
		end	
		connectionsStatus = STAT.CONNECTING
		hubIp = controller_ip
		wssToken = wss_token
		wssUser = wss_user
		-- Send local login command
--		return Send({ method = "hub.info.get" })
		return Send({method="hub.offline.login.ui", params = {user = wss_user, token = wss_token}})
	end	

	local function SetTokensFromStore(token, uuid, expires)
		wss_token = token
		wss_user = uuid
		token_expires = expires
	end
	
	-- Add a specific handler for a given method
	local function RegisterMethodHandler(method, handler)
		if (type(handler) == "function") then
			methodCallbacks[method] = handler 
			return true
		end
		return false, "Handler is not a function"
	end
	local function RegisterBroadcastHandler(message, handler)
		if (type(handler) == "function") then
			broadcastCallbacks[message] = handler 
			return true
		end
		return false, "Handler is not a function"
	end
	local function RegisterErrorHandler(method, handler)
		if (type(handler) == "function") then
			errorCallbacks[method] = handler 
			return true
		end
		return false, "Handler is not a function"
	end
	
	-- Return current connection status
	local function GetConnectionStatus()
		return connectionsStatus
	end

	-- See if we can make use of openLuup.scheduler. However, this maybe needs to be done in luaws module.
	local function StartPoller()
		local R1NAME = "Ezlo_Async_WebSocket_Reciever"
		local RCNAME = "Ezlo_Async_WebSocket_Reconnect"
		local POLL_RATE = 1
    
		local function check_for_data ()
			-- Get data, if more is true, immediately read next chunk.
			local lp_ts = luaws.wslastping(wsconn)
			local res, more, nb = nil, nil, nil
			if os.difftime(os.time(), lp_ts) > 90 then
				log.Debug("No ping received for %d seconds", os.difftime(os.time(),lp_ts))
			else
				res, more, nb = pcall(luaws.wsreceive, wsconn)
				while res and more do
					log.Info(R1NAME .. ", More chunks to receive")
					res, more, nb = pcall(luaws.wsreceive, wsconn)
				end
				if not res then
					log.Error(R1NAME .. ". Error receiving data from Hub, "..tostring(more or ""))
				end
			end	
			-- If more is nil the connection to the Hub is lost. Close and retry.
			if more == nil then
				log.Warning(R1NAME .. "Lost connection to Hub, "..tostring(nb or "").." Try reconnect in "..reconnectRetryInterval)
				Close()
--			luup.call_delay(RCNAME, reconnectRetryInterval, "1")
				socket.sleep(reconnectRetryInterval)
				Reconnect("1")
			else
--				luup.call_delay (R1NAME, POLL_RATE, '')
			end
		end

--		_G[R1NAME] = check_for_data
--		_G[RCNAME] = Reconnect
		check_for_data()
		return true
	end
	
	-- Reconnect to Hub, with up to five retries.
	function Reconnect(retry)
		local retry = tonumber(retry) or 1
		log.Debug("Try to Reconnect, attempt "..retry)
		if retry < maxReconnectRetries then
			local res, msg = Connect(hubIp, wssToken, wssUser)
			if res then
				log.Debug("Connection reopened, login")
--				StartPoller()
			else
				local RCNAME = "Ezlo_Async_WebSocket_Reconnect" -- Is this function
				log.Debug("Could not reconnect, retrying in "..reconnectRetryInterval.." seconds")
--				luup.call_delay (RCNAME, reconnectRetryInterval, tostring(retry + 1))
				socket.sleep(reconnectRetryInterval)
				Reconnect(tostring(retry + 1))
			end
		else
			log.Debug("Could not reconnect after "..maxReconnectRetries.." retries.")
			connectionsStatus = STAT.CONNECT_FAILED
		end
	end

	-- Userfull for Athom that does not like ping, or getting the house mode.
	local function SetPingCommand(cmd)
		if type (cmd) == "table" then
			pingCommand = cmd
		else
			pingCommand = nil
		end
	end
	
	-- Initialize module
	local function Initialize(dbg)
		luaws.Initialize(dbg)
		connectionsStatus = STAT.NO_CONNECTION
	end


	return {
		Initialize = Initialize,
		StartPoller = StartPoller,
		PortalLogin = PortalLogin,
		Connect = Connect,
		SetTokensFromStore = SetTokensFromStore,
		GetConnectionStatus = GetConnectionStatus,
		BAD_PASSWORD = STAT.BAD_PASSWORD,
		TOKEN_EXPIRED = STAT.TOKEN_EXPIRED,
		NO_CONNECTION = STAT.NO_CONNECTION,
		CONNECT_FAILED = STAT.CONNECT_FAILED,
		CONNECTING = STAT.CONNECTING,
		CONNECTED = STAT.CONNECTED,
		IDLE = STAT.IDLE,
		BUSY = STAT.BUSY,
		Send = Send,
		Receive = Receive,
		Ping = Ping,
		SetPingCommand = SetPingCommand,
		Close = Close,
		RegisterMethodHandler = RegisterMethodHandler,
		RegisterBroadcastHandler = RegisterBroadcastHandler,
		RegisterErrorHandler = RegisterErrorHandler
	}
end
local ezlo = ezloAPI()

-- Error handling for specific methods
local function ErrorHandler(method, err, result)
	if method == "hub.offline.login.ui" then
		-- Login error, close connection
		log.Error("Hub login error, closing.")
		log.Error("     Error info %s", json.encode(err))
		ezlo.Close()
	else
		log.Error("Error from hub for method %s", method)
		log.Error("     Error info %s", json.encode(err))
		if result then
			log.Error("     Error result %s", json.encode(result))
		end
	end
	return true
end

-- Handling for UI_broadcasts for specific messages
local function BroadcastHandler(msg_subclass, result)
	log.Debug("Hub broadcast for message %s", msg_subclass)
	log.Debug("      Result %s", json.encode(result))
	return true
end

-- Handlers for method responses
local function MethodHander(method,result)
	if method == "hub.offline.login.ui" then
		log.Debug(json.encode(result))
		log.Debug ("logged on to hub locally.")
		-- Ask for items list for device variables
		log.Debug("Send hub.info.get")
		ezlo.Send({ method = "hub.info.get" })
	elseif method == "hub.info.get" then
		PK_AccessPoint = result.serial
		BuildVersion = result.firmware
		log.Debug("Found Hub serial %s",PK_AccessPoint)
		log.Debug("Found Hub firmware %s",BuildVersion)
--		log.Debug("Send hub.data.list")
--		ezlo.Send({ method = "hub.data.list", params = '{ "devices": null, "items": null, "rooms": null, "scenes": null}' })
--	elseif method == "hub.data.list" then
--		ezlo.Send({ method = "hub.modes.get" })
--	elseif method == "hub.modes.get" then
--		print(json.encode(result))
--		print("Send hub.favorite.list")
--		ezlo.Send({ method = "hub.favorite.list" })
--	elseif method == "hub.favorite.list" then
--		print(json.encode(result))
--		print("Send hub.gateways.list")
--		ezlo.Send({ method = "hub.gateways.list" })
--	elseif method == "hub.gateways.list" then
--		print(json.encode(result))
		print("Send hub.room.list")
		ezlo.Send({ method = "hub.room.list" })
	elseif method == "hub.room.list" then
		print(json.encode(result))
		print("Send hub.scenes.list")
		ezlo.Send({ method = "hub.scenes.list" })
	elseif method == "hub.scenes.list" then
		print(json.encode(result))
		print("Send hub.devices.list")
		ezlo.Send({ method = "hub.devices.list" })
	elseif method == "hub.devices.list" then
--		print(json.encode(result))
--		-- Device list received, ask for item list for details
--		print("Send hub.items.list")
--		ezlo.Send({ method = "hub.items.list" })
--	elseif method == "hub.items.list" then
		-- Device list received, ask for item list for details
--		print(json.encode(result))
--		print("Send hub.device.settings.list")
--		ezlo.Send({ method = "hub.device.settings.list", params = { _id = "5f2965f9124c41108735f58f"} })
--	elseif method == "hub.device.settings.list" then
--		-- Device list received, ask for item list for details
--		log.Debug("hub.data.list result")
		print(json.encode(result))
	else
		print("EzloMessageHandler, response has method ",method) 
	end
	return true
end

-- Init module
local function Initialize()
	json.Initialize()
	log.Initialize("EzloBridge", 10, false)
	ezlo.Initialize(debug_mode)
--	ezlo.RegisterMethodHandler("hub.info.get", MethodHander)
--	ezlo.RegisterMethodHandler("hub.devices.list", MethodHander)
--	ezlo.RegisterMethodHandler("hub.items.list", MethodHander)
--	ezlo.RegisterMethodHandler("hub.room.list", MethodHander)
--	ezlo.RegisterMethodHandler("hub.scenes.list", MethodHander)
	ezlo.RegisterMethodHandler("*", MethodHander)
	ezlo.RegisterErrorHandler("*", ErrorHander)
	ezlo.RegisterBroadcastHandler("*", BroadcastHandler)
end

-- Main routine
Initialize()
-- Get arguments.
if #arg ~= 4 then
	print("Usage : "..arg[0].." HubIP serial userID password")
	return
end	

local ip = arg[1]
local serial = arg[2]
local user_id = arg[3]
local password = arg[4]
local wss_user = ''
local wss_token = ''
local token_expires = 0

-- If we have the logon data for local, no need to get it again.
-- See if we have user and token key stored
local fp = io.open(ezlo_keys_store, "r")
if fp then
	local ts = fp:read("*a")
	fp:close()
	local js_ts = json.decode(ts)
	wss_user = js_ts.user
	wss_token = js_ts.token
	token_expires = js_ts.expires
end
-- No token stored, so logon to portal to obtain and sync with hub
if wss_token == '' or wss_user == '' or token_expires == 0 then
	local stat
	stat, wss_token, wss_user, token_expires = ezlo.PortalLogin(user_id, password, serial)
	if not stat then
		log.Error("Unable to logon to portal %s.", wss_token)
		return
	end	
	-- Write keys to file for reuse
	fp = io.open(ezlo_keys_store, "w")
	fp:write('{"user": "'..wss_user..'","token":"'..wss_token..'","expires":'..token_expires..'}')
	fp:close()
else
	log.Info("Using stored credentials.")
end
print(os.date("Token expires : %c", token_expires))
print("\n")

-- Open web socket connection
local res, msg = ezlo.Connect(ip,wss_token, wss_user)
if not res then
	log.Error("Could not connect to Hub, %s", msg)
	if lfs.attributes(ezlo_keys_store) then
		-- Remove stored key file as they maybe wrong.
		os.remove(ezlo_keys_store)
	end
else
	-- Connection should be ready for commands now
--[[	if ezlo.GetConnectionStatus() == 1 then
        print("\n")
	    print("Send hub.info.get")
		local cmd= '{ "method": "hub.info.get", "id": "hub.info.get" , "params": {} }'
        print(cmd)
		ezlo.Send(cmd)
		-- run for some time, send ping every 30 secs.
		for i = 1, 1000 do
			socket.sleep(1)
			-- Handle incomming data.
			local more, nb = ezlo.Receive()
			if more == nil then
				-- Some error, we are done.
				log.Error("WebSocket receive error %s",nb)
				ezlo.Close()
				break
			end
			-- Read rest of message?
			while more == true do
				print("Receive more chunks to receive")
				more, nb = ezlo.Receive()
			end
			-- Send ping every 30 seconds to keep connection open
--			if math.fmod(i,30) == 0 then
--				print("sending wsping...")
--				ezlo.Ping()
--			end
		end	
	else
		log.Debug("We have some connection error.")
	end
]]
	local cnt = 0
	ezlo.StartPoller()
	while true do
		socket.sleep(1)
		ezlo.StartPoller()
		cnt = cnt + 1
		if cnt == 30 then
--			ezlo.Send({ method = "hub.modes.current.get" })
			cnt = 0
		end
	end
	ezlo.Close()
end
